/**This class provides a variety of useful image manipulation functions
 * for resizing, reformating and recoding images.  Image manipulations
 * in this class are performed using the Java Advanced Imaging library.
 * @author Chris Jarabek (cjjarabe@ucalgary.ca) * 
 */
package org.xenbase.utilities;

import java.awt.image.RenderedImage;
import java.awt.image.renderable.ParameterBlock;
import java.io.ByteArrayOutputStream;

import javax.media.jai.Interpolation;
import javax.media.jai.InterpolationNearest;
import javax.media.jai.JAI;
import javax.media.jai.ParameterBlockJAI;
import javax.media.jai.PlanarImage;

import com.sun.media.jai.codec.ByteArraySeekableStream;
import com.sun.media.jai.codec.ImageCodec;
import com.sun.media.jai.codec.JPEGEncodeParam;
import com.sun.media.jai.codec.PNGEncodeParam;
import com.sun.media.jai.codec.TIFFEncodeParam;

public class AdvImageUtil {

    
	/**Converts images to a limited variety of formats.  
	 * @param toFormat a string denoting the format you want to convert to 'jpg', 'jpeg', 'png', 'tif', tiff'
	 * @param imgBytes the byte array containing the image you want to convert
	 * @return byte array of the newly converted image
	 * @throws Exception
	 * @throws Error
	 */
	public static byte[] convertTo(String toFormat, byte[] imgBytes) throws Exception, Error{
		byte[] retVal = null;
		PlanarImage image;
		try {
			ByteArraySeekableStream bass = new ByteArraySeekableStream(imgBytes);
			image = JAI.create("stream", bass);

			ByteArrayOutputStream os = new ByteArrayOutputStream();
			if ((toFormat.equalsIgnoreCase("jpg")) || toFormat.equalsIgnoreCase("jpeg")) {
				JPEGEncodeParam encodeParam = new JPEGEncodeParam();
				encodeParam.setQuality(0.90F);
				ParameterBlock pb = new ParameterBlock();
				pb.addSource(image);
				pb.add(os);
				pb.add("JPEG");
				pb.add(encodeParam);
				JAI.create("encode", pb, null);
			} else if (toFormat.equalsIgnoreCase("png")) {
				PNGEncodeParam encodeParam = PNGEncodeParam.getDefaultEncodeParam(image);
				ParameterBlock pb = new ParameterBlock();
				pb.addSource(image);
				pb.add(os);
				pb.add("PNG");
				pb.add(encodeParam);
				JAI.create("encode", pb, null);
			} else if (toFormat.equalsIgnoreCase("tif")	|| (toFormat.equalsIgnoreCase("tiff"))) {
				TIFFEncodeParam encodeParam = new TIFFEncodeParam();
				ParameterBlock pb = new ParameterBlock();
				pb.addSource(image);
				pb.add(os);
				pb.add("TIFF");
				pb.add(encodeParam);
				JAI.create("encode", pb, null);
			}
			retVal = os.toByteArray();

		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.convertTo: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.convertTo: " + e.getMessage());
			throw e;
		}
		return retVal;
	}

	/**Determine the image encoding format of the image contained within the passed byte array
	 * @param imgBytes byte array of the image you want to identify.
	 * @return string containing the file type of the image in question 
	 * @throws Exception
	 * @throws Error
	 */
	public static String getFileType(byte[] imgBytes) throws Exception, Error {
		String retVal = "";
		try {
			ByteArraySeekableStream bass = new ByteArraySeekableStream(imgBytes);
			String[] sl = ImageCodec.getDecoderNames(bass);
			retVal = sl[0];
			if (retVal.equalsIgnoreCase("jpeg"))
				retVal = "jpg";
			else if (retVal.equalsIgnoreCase("tiff"))
				retVal = "tif";
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.getFileType: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.getFileType: " + e.getMessage());
			throw e;
		}

		return retVal;
	}

	/**Determines the height of an image.
	 * @param imgBytes image byte array
	 * @return int with the height of the image
	 * @throws Exception
	 * @throws Error
	 */
	public static int getHeight(byte[] imgBytes) throws Exception, Error {
		PlanarImage image;
		int height = 0;
		try {
			ByteArraySeekableStream bass = new ByteArraySeekableStream(imgBytes);
			image = JAI.create("stream", bass);
			height = image.getHeight();
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.getHeight: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.getHeight: " + e.getMessage());
			throw e;
		}
		return height;
	}
	
	/**Determines the width of an image. 
	 * @param imgBytes image byte array
	 * @return int with the width of the image
	 * @throws Exception
	 * @throws Error
	 */
	public static int getWidth(byte[] imgBytes) throws Exception, Error{
		PlanarImage image;
		int width = 0;
		try {
			ByteArraySeekableStream bass = new ByteArraySeekableStream(imgBytes);
			image = JAI.create("stream", bass);
			width = image.getWidth();
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.getWidth: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.getWidth: " + e.getMessage());
			throw e;
		}
		return width;
	}
	
	
	/**Scales an image to the exact height and width provided,
	 * please note, this process will not preserve the aspect ratio of the image.
	 * If you wish to preserve the aspect ratio use scaleToFit.  
	 * @param x int the width you want the resulting image to be.
	 * @param y int the height you want the resulting image to be
	 * @param imgBytes the byte array of the image you want to scale
	 * @return the byte array of the scaled image
	 * @throws Exception
	 * @throws Error
	 */
	public static byte[] scaleTo(int x, int y, byte[] imgBytes) throws Exception, Error {
		PlanarImage image;
		byte[] b = null;
		try {
			ByteArraySeekableStream bass = new ByteArraySeekableStream(imgBytes);
			image = JAI.create("stream", bass);
			Integer Ix = new Integer(x), Iy = new Integer(y);
			Integer width = new Integer(image.getWidth());
			Integer height = new Integer(image.getHeight());
			float scaleX = Ix.floatValue() / width.floatValue();
			float scaleY = Iy.floatValue() / height.floatValue();
			b = scale(scaleX, scaleY, image);
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.scaleTo: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.scaleTo: " + e.getMessage());
			throw e;
		}
		return b;
	}
	
	/**Scales an image to fit the dimensions provided.  Aspect ratio is maintained
	 * therefore it is possible that one of the dimensions of the image will be less 
	 * than the maximum size you specified.  
	 * @param width the maximum width to scale the image to 
	 * @param height the maximum height to scale the image to
	 * @param imgBytes the byte array containing the image
	 * @return byte array with the scaled image
	 * @throws Exception
	 * @throws Error
	 */
	public static byte[] scaleToFit(int width, int height, byte[] imgBytes) throws Exception, Error{
		PlanarImage image;
		try {
			ByteArraySeekableStream bass = new ByteArraySeekableStream(imgBytes);
			image = JAI.create("stream", bass);
			if (image.getWidth() > image.getHeight()) {
				return scaleX((float) width, image);
			} else {
				return scaleY((float) height, image);
			}
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.scaleToFit: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.scaleToFit: " + e.getMessage());
			throw e;
		}
	}
	
	/**Converts a JAI rendered image to a byte array image
	 * @param image JAI Rendered Image
	 * @return byte array of the image in JPEG format
	 * @throws Exception
	 * @throws Error
	 */
	public static byte[] makeImageByteArrayFromRenderedImage(RenderedImage image) throws Exception, Error {
		byte[] b = null;
		try {
			ByteArrayOutputStream retstream = new ByteArrayOutputStream();
			JAI.create("encode", image, retstream, "jpeg");
			b = retstream.toByteArray();
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.makeImageByteArrayFromRenderedImage: " + e.getMessage());
			b = null;
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.makeImageByteArrayFromRenderedImage: " + e.getMessage());
			b = null;
			throw e;
		}
		return b;
	}
	
	 /**Takes a raw byte array and converts it into a JAI RenderedImage
     * that can be used for manipulation (i.e. resizing).
     * @param byte[] b
     * @return RenderedImage
     */
    public static RenderedImage convertImageByteArrayToRenderedImage(byte[] b) throws Exception, Error{
        RenderedImage retVal = null;
        try {
            ByteArraySeekableStream input = new ByteArraySeekableStream(b);
            retVal = JAI.create("stream", input);
        } catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.convertImageByteArrayToRenderedImage: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.convertImageByteArrayToRenderedImage: " + e.getMessage());
			throw e;
		}
        return retVal;
    }	

	private static byte[] scale(float x, float y, PlanarImage image) throws Exception, Error {
		byte[] b = null;
		try{
		    ParameterBlock param = new ParameterBlock();
		    Interpolation interp = Interpolation.getInstance(Interpolation.INTERP_BICUBIC_2);
		    param.addSource(image);
		    param.add(x);
		    param.add(y);
		    param.add(0.0F);
		    param.add(0.0F);
		    param.add(interp);
		    image = JAI.create("scale", param);
		    ByteArrayOutputStream retstream = new ByteArrayOutputStream();
		    JAI.create("encode", image, retstream, "jpeg");
		    b = retstream.toByteArray();
		} catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.scale: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.scale: " + e.getMessage());
			throw e;
		}
		return b;
	}

	private static byte[] scale(float amount, PlanarImage image) throws Exception, Error{
	    byte[] b = null;
	    try{
	        ParameterBlockJAI pb = new ParameterBlockJAI("scale");
	        pb.addSource(image);
	        pb.setParameter("xScale", amount); //x Scale Factor
	        pb.setParameter("yScale", amount); //y Scale Factor
	        pb.setParameter("xTrans", 0.0F); //x Translate amount
	        pb.setParameter("yTrans", 0.0F); //y Translate amount
	        pb.setParameter("interpolation", new InterpolationNearest());
	        image = JAI.create("scale", pb, null);
	        ParameterBlock opb = new ParameterBlock();
	        opb.add(image);
	        ByteArrayOutputStream retstream = new ByteArrayOutputStream();
	        JAI.create("encode", image, retstream, "jpeg");
	        b = retstream.toByteArray();
	    } catch (Exception e) {
			System.out.println("Exception in AdvImageUtil.scale: " + e.getMessage());
			throw e;
		} catch (Error e) {
			System.out.println("Error in AdvImageUtil.scale: " + e.getMessage());
			throw e;
		}
		return b;
	}

	
	private static byte[] scaleX(float width, PlanarImage image) throws Exception, Error{
		float scaleFactor;
		scaleFactor = width / (float) image.getWidth();
		return scale(scaleFactor, image);
	}

	private static byte[] scaleY(float height, PlanarImage image) throws Exception, Error{
		float scaleFactor;
		scaleFactor = height / (float) image.getHeight();
		return scale(scaleFactor, image);
	}

}
